// ===================== Navegación por hash  =====================

const routeViews = {

  panel: [

    document.getElementById('dashboard-content'),

    document.querySelector('#panel > .grid'),

    document.getElementById('latest-events'),

    document.getElementById('bot-status'),

  ],

  tickets: [document.getElementById('tickets')],

  logs: [document.getElementById('logs')],

  comandos: [document.getElementById('comandos')],

  config: [document.getElementById('config')],

};

const navLinks = document.querySelectorAll('.nav a');

const panelRoot = document.getElementById('panel');

const allRouteViews = [...new Set(Object.values(routeViews).flat().filter(Boolean))];

function setRoute(hash){

  const key = (hash||'').replace('#','') || 'panel';

  const target = routeViews[key] ? key : 'panel';

  panelRoot?.removeAttribute('hidden');

  const visible = new Set(routeViews[target] || []);

  allRouteViews.forEach(el=>{

    if(!el) return;

    if(visible.has(el)) el.removeAttribute('hidden'); else el.setAttribute('hidden','');

  });

  navLinks.forEach(a=>{

    const href = a.getAttribute('href')||'';

    if(href.includes('#'+target)) {

      a.classList.add('active');

      a.setAttribute('aria-current','page');

    } else {

      a.classList.remove('active');

      a.removeAttribute('aria-current');

    }

  });

  try { window.scrollTo({ top: 0, behavior: 'instant' }); } catch { window.scrollTo(0,0); }

  if (target === 'panel') {

    try { handleDiscordLinkParams(); } catch (err) { console.error('[ui] discord param handler failed', err); }

    try {

      document.querySelectorAll('.tab')?.forEach(b=>{

        const is = (b.getAttribute('data-tab')||'') === 'metrics';

        b.classList.toggle('active', is);

        b.setAttribute('aria-selected', is ? 'true' : 'false');

      });

      ;['metrics','servicios','webhooks'].forEach(t=>{

        const el = document.getElementById('panel-'+t);

        if (el) el.hidden = (t!=='metrics');

      });

    } catch {}

    loadKpis().catch(()=>{});

    loadLogs('', '#panel #latest-events .table tbody').catch(()=>{});

    loadStatus().catch(()=>{});

  }

  if (target === 'logs')  loadLogs().catch(()=>{});

  if (target === 'tickets') loadTickets().catch(()=>{});

}

function handleDiscordLinkParams(){

  const search = window.location.search || '';

  if (!search.includes('discord=')) return;

  let status = null;

  try {

    const params = new URLSearchParams(search);

    status = params.get('discord');

    params.delete('discord');

    const newQuery = params.toString();

    const next = `${window.location.pathname}${newQuery ? `?${newQuery}` : ''}${window.location.hash || ''}`;

    window.history.replaceState({}, document.title, next);

  } catch (err) {

    console.error('[ui] failed to parse discord param', err);

  }

  if (!status) return;

  switch (status) {

    case 'linked':

      toast('Cuenta de Discord vinculada correctamente', 'success');

      break;

    case 'conflict':

      toast('Esa cuenta de Discord ya está vinculada a otro usuario', 'error');

      break;

    case 'no-session':

      toast('Inicia sesión en el panel antes de conectar Discord', 'error');

      break;

    default:

      toast('No se pudo vincular Discord', 'error');

      break;

  }

}

window.addEventListener('hashchange', ()=> setRoute(location.hash));

setRoute(location.hash);



// =============================== Tabs  ===============================

const tabButtons = document.querySelectorAll('.tab');

tabButtons.forEach(btn=>{

  btn.addEventListener('click',()=>{

    tabButtons.forEach(b=>b.classList.remove('active'));

    btn.classList.add('active');

    const k = btn.dataset.tab;

    ["metrics","servicios","webhooks"].forEach(t=>{

      const el = document.getElementById('panel-'+t);

      if(el) el.hidden = t!==k;

    });

  })

});



// ========================= Buscador Ctrl+K  =========================

const q = document.getElementById('q');

window.addEventListener('keydown', (e)=>{

  if((e.ctrlKey || e.metaKey) && e.key.toLowerCase()==='k'){

    e.preventDefault(); q?.focus();

  }

});



// ============================= Modal tickets  =============================

const modal = document.getElementById('modal-ticket');

document.querySelectorAll('[data-open-modal="ticket"]').forEach(btn=>{

  btn.addEventListener('click', ()=> modal?.showModal());

});

modal?.querySelector('[data-close]')?.addEventListener('click', ()=> modal?.close());



// =============================== Toast helper  ===============================

function toast(msg, kind){

  const host = document.getElementById('toast');

  if(!host) return;

  const el = document.createElement('div');

  el.className = 'item' + (kind ? ` ${kind}` : '');

  el.textContent = msg;

  host.appendChild(el);

  setTimeout(()=>{ el.style.opacity=.0; el.style.transform='translateY(6px)'; }, 2600);

  setTimeout(()=>{ host.removeChild(el); }, 3200);

}



// ================================ Acciones demo (tu código) ================================

document.getElementById('btnSync')?.addEventListener('click', ()=> toast('Sincronización iniciada…'));

document.getElementById('btnDeploy')?.addEventListener('click', ()=> toast('Desplegando…'));



// =============================== Tema (tu código) ===============================



// ---- Theme selector (modal + persist) ----

const THEME_KEY = 'panel:theme';

function applyTheme(theme, opts={persist:true}){

  const t = (theme||'').toLowerCase();

  const allowed = ['light','dark','fire'];

  const final = allowed.includes(t) ? t : 'dark';

  document.documentElement.setAttribute('data-theme', final);

  document.documentElement.style.colorScheme = (final === 'light') ? 'light' : 'dark';

  if (opts.persist) try { localStorage.setItem(THEME_KEY, final); } catch {}

  updateActiveThemeInModal(final);

}

function getStoredTheme(){ try { return localStorage.getItem(THEME_KEY) || ''; } catch { return ''; } }

function getPreferredTheme(){

  const stored = getStoredTheme();

  if (stored) return stored;

  const prefersLight = window.matchMedia && window.matchMedia('(prefers-color-scheme: light)').matches;

  return prefersLight ? 'light' : 'dark';

}

function updateActiveThemeInModal(current){

  document.querySelectorAll('[data-theme-option]')?.forEach(card=>{

    const v = card.getAttribute('data-theme-option');

    if (v === current) card.classList.add('active'); else card.classList.remove('active');

  });

}

function initThemeUI(){

  const old = document.getElementById('toggleTheme');

  let toggleButton = old;

  // Replace node to remove earlier click listeners

  if (old) { const clone = old.cloneNode(true); old.replaceWith(clone); toggleButton = clone; }

  const dlg = document.getElementById('modal-theme');

  const btnClose = document.getElementById('theme-close');

  const btnApply = document.getElementById('theme-apply');

  const cards = document.querySelectorAll('[data-theme-option]');

  const getCurrentTheme = () => document.documentElement.getAttribute('data-theme') || getPreferredTheme();

  const closeModal = () => { try { dlg?.close(); } catch { if (dlg) dlg.open = false; } };

  let pending = getPreferredTheme();

  let openedTheme = pending;

  let applied = false;

  toggleButton?.addEventListener('click', ()=>{

    openedTheme = getCurrentTheme();

    pending = openedTheme;

    applied = false;

    updateActiveThemeInModal(openedTheme);

    try { dlg?.showModal(); } catch { if (dlg) dlg.open = true; }

  });

  btnClose?.addEventListener('click', ()=> closeModal());

  dlg?.addEventListener('close', ()=>{

    if (!applied) applyTheme(openedTheme, { persist: false });

    pending = getCurrentTheme();

  });

  cards?.forEach(card=>{

    card.addEventListener('click', ()=>{

      pending = card.getAttribute('data-theme-option') || 'dark';

      applyTheme(pending, { persist: false });

    });

  });

  btnApply?.addEventListener('click', ()=>{

    applied = true;

    applyTheme(pending);

    toast('Tema: ' + (pending==='light'?'claro': pending==='dark'?'oscuro':'fuego'));

    closeModal();

  });

  updateAdminCreateRoleOptions();

}



// =========================================

// Acciones demo

// =========================================

document.querySelectorAll('[data-action]')?.forEach(b=>{

  b.addEventListener('click', ()=> toast('Acción: '+b.dataset.action));

});



// =====================================================

// ===== A PARTIR DE AQUÍ: CONEXIÓN CON EL PANEL API ====

// =====================================================



// ==================================== Helpers REST ====================================

function computeApiBase() {

  if (typeof window.API_BASE !== 'undefined' && window.API_BASE) return window.API_BASE.replace(/\/+$/, '');

  const meta = document.querySelector('meta[name="base-path"]');

  const basePath = (meta?.content || '').replace(/\/+$/, '').replace(/^\/+/, '');

  if (basePath) return window.location.origin + '/' + basePath;

  // Heurística: tomar el primer segmento del path si existe (p.ej. /panel)

  const seg = (window.location.pathname || '/').split('/').filter(Boolean)[0];

  return window.location.origin + (seg ? '/' + seg : '');

}

let API_BASE = computeApiBase();

// Debug banner eliminado



async function fetchJSON(path, opts = {}) {

  const res = await fetch(API_BASE + path, {

    credentials: 'include',

    headers: { 'Content-Type': 'application/json', ...(opts.headers||{}) },

    method: opts.method || 'GET',

    body: opts.body ? JSON.stringify(opts.body) : undefined

  });

  if (!res.ok) {

    let text = ''; try { text = await res.text(); } catch {}

    throw new Error(`HTTP ${res.status} ${path}${text ? ` - ${text}` : ''}`);

  }

  return res.json();

}



// ========================= Login UI =========================

// Re-define refreshAuthUI with avatar support and robust hidden toggling

async function refreshAuthUI() {

  const loginView = document.getElementById('login-screen');

  const appLayout = document.getElementById('app');

  const connectBtn = document.getElementById('btn-connect-discord');

  const userPill = document.getElementById('user-pill');

  const postLoginCard = document.getElementById('post-login-card');

  const errorEl = document.getElementById('panel-login-error');



  const setLoggedOut = () => {

    appLayout?.setAttribute('hidden', '');

    loginView?.removeAttribute('hidden');

    const loginInput = loginView?.querySelector("input[name='username']");

    loginInput?.focus();

    userPill?.setAttribute('hidden', '');

    postLoginCard?.setAttribute('hidden', '');

    connectBtn?.setAttribute('hidden', '');

    const connectLabel = connectBtn?.querySelector('.connect-label');

    if (connectLabel) connectLabel.textContent = 'Conectar Discord';

    connectBtn?.classList.remove('connected');

    if (PANEL_POLLERS_INTERVAL) { clearInterval(PANEL_POLLERS_INTERVAL); PANEL_POLLERS_INTERVAL = null; }

    if (PANEL_TICKETS_INTERVAL) { clearInterval(PANEL_TICKETS_INTERVAL); PANEL_TICKETS_INTERVAL = null; }

    PANEL_POLLERS_STARTED = false;

    window.CURRENT_USER = null;

    errorEl?.setAttribute('hidden', '');

    try { applyTheme('dark', { persist: false }); } catch {}

  };



  try {

    const { user } = await fetchJSON('/api/me');

    const mapped = { ...user };

    const fallbackLogin = Number(user?.lastLoginAt ?? user?.updatedAt ?? user?.createdAt ?? 0);

    mapped.sessionStartedAt = Date.now();

    if ((mapped.lastLoginAt == null || Number.isNaN(Number(mapped.lastLoginAt))) && fallbackLogin) mapped.lastLoginAt = fallbackLogin;

    window.CURRENT_USER = mapped;

    loginView?.setAttribute('hidden', '');

    appLayout?.removeAttribute('hidden');
    try { setRoute(window.location.hash || '#panel'); } catch (err) { console.error('[ui] failed to apply route after login', err); }


    if (connectBtn) {

      connectBtn.removeAttribute('hidden');

      const linked = Boolean(mapped?.discordId);

      const connectLabel = connectBtn.querySelector('.connect-label');

      if (connectLabel) connectLabel.textContent = linked ? 'Discord conectado' : 'Conectar Discord';

      connectBtn.classList.toggle('connected', linked);

    }

    if (userPill) {

      userPill.removeAttribute('hidden');

      const nameEl = userPill.querySelector('.user-name');

      const roleEl = userPill.querySelector('.user-role');

      const avatarEl = userPill.querySelector('.user-avatar');

      const role = mapped?.role || mapped?.rank || '';

      const roleDisplay = role ? ` (${role.charAt(0).toUpperCase() + role.slice(1)})` : '';

      if (nameEl) nameEl.textContent = mapped?.username || '';

      if (roleEl) roleEl.textContent = roleDisplay;

      if (avatarEl) {

        const avatarUrl = mapped?.avatarUrl || (mapped?.avatar ? `https://cdn.discordapp.com/avatars/${mapped.id}/${mapped.avatar}.png?size=64` : 'https://cdn.discordapp.com/embed/avatars/0.png');

        avatarEl.src = avatarUrl;

        avatarEl.alt = mapped?.username || 'user';

      }

    }

    if (postLoginCard) postLoginCard.removeAttribute('hidden');

    if (errorEl) errorEl.setAttribute('hidden', '');

    return mapped;

  } catch (err) {

    setLoggedOut();

    return null;

  }

}





let PANEL_POLLERS_STARTED = false;

let PANEL_POLLERS_INTERVAL = null;

let PANEL_TICKETS_INTERVAL = null;



async function onAuthenticated(user) {

  if (!user) return;

  try { applyTheme(getStoredTheme() || 'dark', { persist:false }); } catch {}

  try { applyPermissions(user); } catch {}

  try {

    await Promise.all([

      loadKpis(),

      loadLogs('', '#panel #latest-events .table tbody'),

      loadStatus(),

    ]);

  } catch (err) {

    console.error(err);

    toast('No se pudieron cargar los datos iniciales', 'error');

  }

  if (!PANEL_POLLERS_STARTED) {

    initLive();

    PANEL_POLLERS_INTERVAL = setInterval(() => {

      loadKpis().catch(()=>{});

      loadStatus().catch(()=>{});

    }, 30_000);

    PANEL_TICKETS_INTERVAL = setInterval(async () => {

      try { await fetchJSON('/api/tickets/sync', { method: 'POST' }); } catch {}

      try { await loadTickets(); } catch {}

    }, 60_000);

    PANEL_POLLERS_STARTED = true;

  }

}



function initPanelLogin() {

  const form = document.getElementById('panel-login-form');

  if (!form) return;

  const errorEl = document.getElementById('panel-login-error');

  const submitBtn = document.getElementById('panel-login-submit');

  const userInput = form.querySelector("input[name='username']");

  userInput?.focus();

  form.addEventListener('input', () => errorEl?.setAttribute('hidden', ''));

  form.addEventListener('submit', async (event) => {

    event.preventDefault();

    const formData = new FormData(form);

    const username = String(formData.get('username') || '').trim();

    const password = String(formData.get('password') || '');

    if (!username || !password) {

      if (errorEl) {

        errorEl.textContent = 'Completa usuario y contraseña';

        errorEl.removeAttribute('hidden');

      }

      return;

    }

    submitBtn?.setAttribute('disabled', '');

    if (errorEl) errorEl.setAttribute('hidden', '');

    try {

      await fetchJSON('/api/auth/login', { method: 'POST', body: { username, password } });

      form.reset();

      const user = await refreshAuthUI();

      if (user) await onAuthenticated(user);

    } catch (err) {

      console.error('[login] failed', err);

      if (errorEl) {

        const message = String(err?.message || '').includes('401') ? 'Credenciales inválidas' : 'No se pudo iniciar sesión';

        errorEl.textContent = message;

        errorEl.removeAttribute('hidden');

      }

    } finally {

      submitBtn?.removeAttribute('disabled');

    }

  });

}



const ADMIN_STATE = {

  accounts: [],

  roles: [],

  defaults: {},

  availableRoles: [],

};



const ADMIN_PERMISSIONS = [

  { key: 'viewPanel', label: 'Ver panel principal', readonly: true },

  { key: 'viewTickets', label: 'Ver tickets' },

  { key: 'viewLogs', label: 'Ver logs' },

  { key: 'viewCommands', label: 'Ver comandos' },

  { key: 'viewConfig', label: 'Ver configuración' },

  { key: 'manageUsers', label: 'Gestionar usuarios del panel' },

  { key: 'manageRoles', label: 'Editar permisos' },

];



function generateFrontendAccountCode(length = 6) {

  const alphabet = 'ABCDEFGHJKLMNPQRSTUVWXYZ23456789';

  let code = '';

  for (let i = 0; i < length; i += 1) {

    code += alphabet[Math.floor(Math.random() * alphabet.length)];

  }

  return code;

}



function formatDateTime(ms) {

  if (!ms) return '-';

  try {

    return new Date(Number(ms)).toLocaleString();

  } catch {

    return '-';

  }

}



function setAdminMessage(elOrId, text, kind = 'info') {

  const el = typeof elOrId === 'string' ? document.getElementById(elOrId) : elOrId;

  if (!el) return;

  if (!text) {

    el.textContent = '';

    el.setAttribute('hidden', '');

    el.classList.remove('info', 'success', 'error');

    return;

  }

  el.textContent = text;

  el.removeAttribute('hidden');

  el.classList.remove('info', 'success', 'error');

  if (kind) el.classList.add(kind);

}






function getAvailableRoles(){

  const roles = ADMIN_STATE?.availableRoles;

  if (Array.isArray(roles) && roles.length) return roles;

  return ['superadmin', 'admin', 'staff', 'mod', 'viewer'];

}

function populateRoleSelect(select, currentRole){

  if (!select) return;

  const roles = getAvailableRoles();

  const normalized = String(currentRole || '').toLowerCase();

  const fallback = String(select.value || '').toLowerCase();

  const target = normalized || fallback || roles[0] || '';

  select.innerHTML = '';

  roles.forEach(role => {

    const option = document.createElement('option');

    option.value = role;

    option.textContent = role.charAt(0).toUpperCase() + role.slice(1);

    option.selected = role === target;

    select.appendChild(option);

  });

  if (!select.value && target) select.value = target;

}

function updateAdminCreateRoleOptions(){

  const createSelect = document.querySelector("#admin-user-create select[name='role']");

  if (!createSelect) return;

  populateRoleSelect(createSelect, createSelect.value);

}

function renderAdminAccounts() {

  const tbody = document.getElementById('admin-users-table');

  if (!tbody) return;

  tbody.innerHTML = '';

  const list = Array.isArray(ADMIN_STATE.accounts) ? ADMIN_STATE.accounts.slice() : [];

  if (!list.length) {

    const row = document.createElement('tr');

    const cell = document.createElement('td');

    cell.colSpan = 5;

    cell.className = 'empty-row';

    cell.textContent = 'No hay usuarios registrados';

    row.appendChild(cell);

    tbody.appendChild(row);

    updateAdminCreateRoleOptions();

    return;

  }

  list.sort((a, b) => {

    const nameA = (a?.username || '').toLocaleLowerCase('es');

    const nameB = (b?.username || '').toLocaleLowerCase('es');

    return nameA.localeCompare(nameB);

  });

  list.forEach(acc => {

    const tr = document.createElement('tr');

    const accountId = acc?.id ?? acc?.accountId ?? acc?.accountID;

    if (accountId != null && accountId !== '') tr.dataset.accountId = String(accountId);

    if (acc?.discordId) tr.dataset.discordId = String(acc.discordId);

    const codeCell = document.createElement('td');

    const codeWrap = document.createElement('div');

    codeWrap.className = 'admin-code';

    const codeEl = document.createElement('code');

    codeEl.textContent = acc?.accountCode || '-';

    const copyBtn = document.createElement('button');

    copyBtn.className = 'btn';

    copyBtn.setAttribute('data-action', 'copy-code');

    copyBtn.setAttribute('title', 'Copiar ID');

    copyBtn.textContent = 'Copiar';

    codeWrap.appendChild(codeEl);

    codeWrap.appendChild(copyBtn);

    codeCell.appendChild(codeWrap);

    tr.appendChild(codeCell);

    const infoCell = document.createElement('td');

    const nameDiv = document.createElement('div');

    nameDiv.className = 'admin-user-name';

    nameDiv.textContent = acc?.username || '-';

    const subDiv = document.createElement('div');

    subDiv.className = 'admin-user-sub';

    subDiv.textContent = acc?.discordId ? `Discord: ${acc.discordId}` : 'Sin vincular';

    infoCell.appendChild(nameDiv);

    infoCell.appendChild(subDiv);

    tr.appendChild(infoCell);

    const roleCell = document.createElement('td');

    const select = document.createElement('select');

    select.setAttribute('data-role-select', '');

    populateRoleSelect(select, (acc?.role || '').toString().toLowerCase());

    roleCell.appendChild(select);

    tr.appendChild(roleCell);

    const createdCell = document.createElement('td');

    createdCell.textContent = formatDateTime(acc?.createdAt);

    tr.appendChild(createdCell);

    const actionsCell = document.createElement('td');

    actionsCell.className = 'actions';

    const saveBtn = document.createElement('button');

    saveBtn.className = 'btn brand';

    saveBtn.setAttribute('data-action', 'save-role');

    saveBtn.textContent = 'Guardar';

    const resetBtn = document.createElement('button');

    resetBtn.className = 'btn';

    resetBtn.setAttribute('data-action', 'reset-password');

    resetBtn.textContent = 'Reset pass';

    actionsCell.appendChild(saveBtn);

    actionsCell.appendChild(resetBtn);

    tr.appendChild(actionsCell);

    tbody.appendChild(tr);

  });

  updateAdminCreateRoleOptions();

}

async function loadAdminAccounts() {

  const { accounts } = await fetchJSON('/api/admin/accounts');

  ADMIN_STATE.accounts = accounts || [];

  renderAdminAccounts();

  setAdminMessage('admin-user-message');

}



// ========================================= KPIs =========================================

async function loadKpis() {

  const d = await fetchJSON('/api/kpis');

  setText('kpi-members', d.events24h ?? '-');

  setText('kpi-members-delta', '');

  setText('kpi-users',   (d.users ?? 0).toLocaleString());

  setText('kpi-tickets', d.ticketsOpen ?? '-');

  setText('kpi-errors',  d.errors24h ?? '-');

}

function setText(id, value){

  const el = document.getElementById(id);

  if (el) el.textContent = value;

}



// ======================================== Status ========================================

function renderStatusMetrics(m = {}){

  if (m.wsPing != null) {

    setText('st-latency', `${m.wsPing} ms`);

    setBadgeState('st-latency', m.wsPing, { ok: v=>v<100, warn: v=>v<250 });

  }

  if (m.memMB != null) {

    setText('st-mem', `${m.memMB} MB`);

    setBadgeState('st-mem', m.memMB, { ok: v=>v<400, warn: v=>v<800 });

  }

  if (m.cpuPct != null) {

    setText('st-cpu', `${m.cpuPct}`);

    setTextState('st-cpu', m.cpuPct, { ok: v=>v<30, warn: v=>v<70 });

  }

  if (m.usersCached != null) setText('st-users', `${m.usersCached}`);

  if (m.uptimeSec != null) setText('st-uptime', formatUptime(m.uptimeSec));

  if (Array.isArray(m.shards)) {

    m.shards.forEach((shard, idx) => {

      const id = shard?.id ?? idx;

      const el = document.getElementById(`st-shard-${id}`);

      if (!el) return;

      const status = shard?.status || shard?.state || (shard?.online === false ? 'Offline' : 'Online');

      el.textContent = status;

    });

  }

}



function renderAdminRoles() {

  const container = document.getElementById('admin-roles-container');

  if (!container) return;

  container.innerHTML = '';

  ADMIN_STATE.roles.forEach(roleInfo => {

    const role = roleInfo.role;

    const card = document.createElement('div');

    card.className = 'admin-role-card';

    card.dataset.role = role;

    const effective = roleInfo.effective || {};

    card.innerHTML = `

      <h4>${role.charAt(0).toUpperCase() + role.slice(1)}</h4>

      <div class="admin-role-perms">${ADMIN_PERMISSIONS.map(perm => {

        const checked = effective[perm.key] ?? false;

        const disabled = perm.readonly ? 'disabled' : '';

        return `<label><input type="checkbox" data-permission="${perm.key}" ${checked ? 'checked' : ''} ${disabled}/> ${perm.label}</label>`;

      }).join('')}</div>

      <div class="admin-role-actions">

        <button class="btn brand" data-action="save-role">Guardar</button>

        <button class="btn" data-action="reset-role">Restablecer</button>

      </div>`;

    container.appendChild(card);

  });

}



async function loadAdminRoles() {

  const { roles, defaults } = await fetchJSON('/api/admin/roles');

  ADMIN_STATE.roles = roles || [];

  ADMIN_STATE.defaults = defaults || {};

  const availableSet = new Set();

  Object.keys(ADMIN_STATE.defaults || {}).forEach(role => { if (role) availableSet.add(String(role).toLowerCase()); });

  (ADMIN_STATE.roles || []).forEach(item => {

    if (item?.role) availableSet.add(String(item.role).toLowerCase());

  });

  ADMIN_STATE.availableRoles = Array.from(availableSet).filter(Boolean).sort((a, b) => a.localeCompare(b, 'es', { sensitivity: 'base' }));

  renderAdminRoles();

  renderAdminAccounts();

  setAdminMessage('admin-roles-message');

  updateAdminCreateRoleOptions();

}



function switchAdminSection(target) {

  const tabs = document.querySelectorAll('.admin-tab');

  const sections = document.querySelectorAll('.admin-section');

  tabs.forEach(tab => {

    const isActive = tab.getAttribute('data-admin-tab') === target;

    tab.classList.toggle('active', isActive);

    tab.setAttribute('aria-selected', isActive ? 'true' : 'false');

  });

  sections.forEach(section => {

    const match = section.getAttribute('data-admin-section') === target;

    section.hidden = !match;

  });

  if (target === 'roles') {

    loadAdminRoles().catch(()=>{});

  }

  if (target === 'discord') {

    refreshDiscordRanks().catch(()=>{});

  }

}



function initAdminModal() {

  if (window.__ADMIN_MODAL_READY) return;

  window.__ADMIN_MODAL_READY = true;

  const adminDlg = document.getElementById('modal-admin');

  const tabs = document.querySelectorAll('.admin-tab');

  tabs.forEach(tab => {

    tab.addEventListener('click', () => {

      const target = tab.getAttribute('data-admin-tab');

      switchAdminSection(target);

    });

  });

  const closeButtons = [document.getElementById('admin-close'), document.getElementById('admin-ok')];

  closeButtons.forEach(btn => btn?.addEventListener('click', () => {

    try { adminDlg?.close(); } catch { if (adminDlg) adminDlg.open = false; }

  }));

  adminDlg?.addEventListener('close', () => {

    setAdminMessage('admin-user-message');

    setAdminMessage('admin-roles-message');

    setAdminMessage('admin-ranks-message');

  });

  const form = document.getElementById('admin-user-create');

  const codeInput = form?.querySelector("input[name='accountCode']");

  const generateBtn = document.getElementById('admin-user-generate');

  generateBtn?.addEventListener('click', () => {

    if (codeInput) codeInput.value = generateFrontendAccountCode();

  });

  document.getElementById('admin-users-refresh')?.addEventListener('click', async () => {

    try {

      await loadAdminAccounts();

      setAdminMessage('admin-user-message', 'Usuarios actualizados', 'success');

    } catch {

      setAdminMessage('admin-user-message', 'No se pudieron cargar usuarios', 'error');

    }

  });

  document.getElementById('admin-roles-refresh')?.addEventListener('click', async () => {

    try {

      await loadAdminRoles();

      setAdminMessage('admin-roles-message', 'Permisos actualizados', 'success');

    } catch {

      setAdminMessage('admin-roles-message', 'No se pudieron cargar permisos', 'error');

    }

  });

  const accountsTable = document.getElementById('admin-users-table');

  accountsTable?.addEventListener('click', async (event) => {

    const btn = event.target.closest('button');

    if (!btn) return;

    const action = btn.getAttribute('data-action');

    const row = btn.closest('tr');

    const id = Number(row?.dataset.accountId || '');

    if (!Number.isFinite(id)) return;

    if (action === 'copy-code') {

      const code = row?.querySelector('code')?.textContent || '';

      try {

        await navigator.clipboard.writeText(code);

        setAdminMessage('admin-user-message', `ID ${code} copiado`, 'success');

      } catch {

        setAdminMessage('admin-user-message', 'No se pudo copiar el ID', 'error');

      }

      return;

    }

    if (action === 'save-role') {

      const select = row?.querySelector('select[data-role-select]');

      const role = select?.value || '';

      const codeText = row?.querySelector('code')?.textContent?.trim();

      const body = { role };

      if (codeText && codeText !== '-') body.accountCode = codeText;

      try {

        await fetchJSON(`/api/admin/accounts/${id}`, { method: 'PUT', body });

        setAdminMessage('admin-user-message', 'Rol actualizado', 'success');

        await loadAdminAccounts();

      } catch (err) {

        console.error('[admin] save role error', err);

        setAdminMessage('admin-user-message', 'No se pudo actualizar el rol', 'error');

      }

      return;

    }

    if (action === 'reset-password') {

      const newPass = window.prompt('Nueva contraseña para el usuario:');

      if (!newPass) return;

      try {

        await fetchJSON(`/api/admin/accounts/${id}/password`, { method: 'POST', body: { password: newPass } });

        setAdminMessage('admin-user-message', 'Contraseña actualizada', 'success');

      } catch (err) {

        console.error('[admin] reset password error', err);

        setAdminMessage('admin-user-message', 'No se pudo actualizar la contraseña', 'error');

      }

    }

  });

  const rolesContainer = document.getElementById('admin-roles-container');

  rolesContainer?.addEventListener('click', async (event) => {

    const btn = event.target.closest('button');

    if (!btn) return;

    const action = btn.getAttribute('data-action');

    const card = btn.closest('.admin-role-card');

    const role = card?.dataset.role;

    if (!role) return;

    if (action === 'save-role') {

      const checkboxes = card.querySelectorAll('input[data-permission]');

      const payload = {};

      checkboxes.forEach(cb => {

        const key = cb.getAttribute('data-permission');

        payload[key] = cb.checked;

      });

      try {

        await fetchJSON(`/api/admin/roles/${role}`, { method: 'PUT', body: { permissions: payload } });

        setAdminMessage('admin-roles-message', 'Permisos guardados', 'success');

        await loadAdminRoles();

      } catch (err) {

        console.error('[admin] save permissions error', err);

        setAdminMessage('admin-roles-message', 'No se pudieron guardar los permisos', 'error');

      }

      return;

    }

    if (action === 'reset-role') {

      const defaults = ADMIN_STATE.defaults?.[role] || {};

      card.querySelectorAll('input[data-permission]').forEach(cb => {

        const key = cb.getAttribute('data-permission');

        cb.checked = Boolean(defaults[key]);

      });

    }

  });

  const roleCreateForm = document.getElementById('admin-role-create');

  roleCreateForm?.addEventListener('submit', async (event) => {

    event.preventDefault();

    const input = roleCreateForm.querySelector("input[name='role']");

    const raw = String(input?.value || '').trim();

    if (!raw) {

      setAdminMessage('admin-roles-message', 'Ingresa un nombre de rol', 'error');

      return;

    }

    const normalized = raw.toLowerCase();

    if (!/^[a-z0-9_-]{3,32}$/.test(normalized)) {

      setAdminMessage('admin-roles-message', 'El rol debe tener entre 3 y 32 caracteres y solo letras, numeros, guiones o guiones bajos', 'error');

      return;

    }

    if (getAvailableRoles().includes(normalized)) {

      setAdminMessage('admin-roles-message', 'Ese rol ya existe', 'error');

      return;

    }

    try {

      await fetchJSON(`/api/admin/roles/${encodeURIComponent(normalized)}`, { method: 'PUT', body: { permissions: {} } });

      if (input) input.value = '';

      await loadAdminRoles();

      await loadAdminAccounts();

      setAdminMessage('admin-roles-message', 'Rol creado correctamente', 'success');

    } catch (err) {

      console.error('[admin] create role error', err);

      setAdminMessage('admin-roles-message', 'No se pudo crear el rol', 'error');

    }

  });


  const createForm = document.getElementById('admin-user-create');

  createForm?.addEventListener('submit', async (event) => {

    event.preventDefault();

    const formData = new FormData(createForm);

    const raw = Object.fromEntries(formData.entries());

    const payload = {

      accountCode: String(raw.accountCode || '').trim() || generateFrontendAccountCode(),

      username: String(raw.username || '').trim(),

      password: String(raw.password || ''),

      role: String(raw.role || '').trim(),

      avatarUrl: String(raw.avatarUrl || '').trim(),

    };

    if (!payload.username || !payload.password || !payload.role) {

      setAdminMessage('admin-user-message', 'Completa usuario, contraseña y rol', 'error');

      return;

    }

    if (!payload.avatarUrl) delete payload.avatarUrl;

    try {

      await fetchJSON('/api/admin/accounts', { method: 'POST', body: payload });

      setAdminMessage('admin-user-message', 'Usuario creado correctamente', 'success');

      form.reset();

      updateAdminCreateRoleOptions();

      const codeInput = form.querySelector("input[name='accountCode']");

      if (codeInput) codeInput.value = generateFrontendAccountCode();

      await loadAdminAccounts();

    } catch (err) {

      console.error('[admin] create user error', err);

      setAdminMessage('admin-user-message', 'No se pudo crear el usuario', 'error');

    }

  });

}

async function loadStatus(){

  const s = await fetchJSON('/api/status');

  renderStatusMetrics(s?.metrics || {});

}

function formatUptime(sec){ const d=Math.floor(sec/86400),h=Math.floor((sec%86400)/3600),m=Math.floor((sec%3600)/60); return [d?d+'d':null,h?h+'h':null,(sec%60)+'s'].filter(Boolean).join(' '); }



// Aplica clases de color a un chip .badge-pill que contiene el elemento con id dado

function setBadgeState(valueElId, value, rules){

  const el = document.getElementById(valueElId);

  if (!el) return;

  const chip = el.closest('.badge-pill');

  if (!chip) return;

  chip.classList.remove('pill-ok','pill-warn','pill-err');

  let cls = 'pill-err';

  try {

    if (rules?.ok && rules.ok(value)) cls = 'pill-ok';

    else if (rules?.warn && rules.warn(value)) cls = 'pill-warn';

  } catch {}

  chip.classList.add(cls);

}



// Aplica clases de color a un <b id="..."> dentro de listas

function setTextState(valueElId, value, rules){

  const el = document.getElementById(valueElId);

  if (!el) return;

  el.classList.remove('text-ok','text-warn','text-err');

  let cls = 'text-err';

  try {

    if (rules?.ok && rules.ok(value)) cls = 'text-ok';

    else if (rules?.warn && rules.warn(value)) cls = 'text-warn';

  } catch {}

  el.classList.add(cls);

}



// Oculta/ muestra secciones según permisos

function applyPermissions(user){

  const role = String(user?.role || user?.rank || '').toLowerCase();

  const perms = user?.permissions || {};

  const nav = document.querySelector('.nav');

  const navConfig = [

    ['tickets', perms.viewTickets ?? ['staff','mod','admin','superadmin'].includes(role)],

    ['logs', perms.viewLogs ?? ['staff','mod','admin','superadmin'].includes(role)],

    ['comandos', perms.viewCommands ?? ['admin','superadmin'].includes(role)],

    ['config', perms.viewConfig ?? ['admin','superadmin'].includes(role)],

  ];

  navConfig.forEach(([route, allowed])=>{

    const link = nav?.querySelector(`[data-route="${route}"]`);

    if (!link) return;

    if (!allowed) link.setAttribute('hidden',''); else link.removeAttribute('hidden');

  });

  const adminLink = document.getElementById('btn-admin');

  const canManage = perms.manageUsers ?? ['admin','superadmin'].includes(role);

  if (!canManage) adminLink?.setAttribute('hidden',''); else adminLink?.removeAttribute('hidden');

}



// ======================================== Logs ========================================

const LOG_MAX_ROWS = 10;

function trimTableBodyRows(tbody, max = LOG_MAX_ROWS){

  if (!tbody) return;

  const rows = tbody.querySelectorAll('tr');

  for (let i = max; i < rows.length; i++) {

    rows[i].remove();

  }

}

async function loadLogs(guildId = '', containerSelector) {

  const rows = await fetchJSON('/api/logs?limit=10');

  const tbody = (containerSelector ? document.querySelector(containerSelector) : null)

            || document.querySelector('#logs .table tbody')

            || document.getElementById('tbl-logs')

            || document.querySelector('#logs tbody');

  if (!tbody) return;

  tbody.innerHTML = '';

  rows.forEach(l=>{

    const tr = document.createElement('tr');

    const sev = (l.severity || 'info').toLowerCase();

    const className = sev === 'error' ? 'err' : (sev === 'warn' ? 'warn' : 'ok');

    const sevText = sev.charAt(0).toUpperCase() + sev.slice(1);

    tr.innerHTML = `

      <td>${new Date(l.ts).toLocaleString()}</td>

      <td>${l.origin||'-'}</td>

      <td>${l.event||'-'}</td>

      <td><span class="status"><span class="dot ${className}"></span>${sevText}</span></td>`;

    tbody.appendChild(tr);

  });

  trimTableBodyRows(tbody, LOG_MAX_ROWS);

}



// ======================================= Tickets =======================================

async function loadTickets() {

  const rows = await fetchJSON('/api/tickets');

  const tbody = document.querySelector('#tickets .table tbody')

             || document.getElementById('tbl-tickets')

             || document.querySelector('#tickets tbody');

  if (!tbody) return;

  tbody.innerHTML = '';

  rows.forEach(t => {

    const tr = document.createElement('tr');

    tr.innerHTML = `

      <td>#${t.id}</td>

      <td><code>${t.userId || '-'}</code></td>

      <td>${t.assignedId ? `<code>${t.assignedId}</code>` : '-'}</td>

      <td>${t.category || '-'}</td>

      <td>${t.status || '-'}</td>

    `;

    tbody.appendChild(tr);

  });

}



// =================================== Live (opcional) ===================================

function initLive() {

  if (typeof io !== 'function') { console.warn('[live] Socket.IO client no está cargado'); return; }

  const basePath = (new URL(API_BASE)).pathname.replace(/\/+$/, '');

  const socket = io(`${API_BASE}/live`, { withCredentials: true, path: `${basePath || ''}/socket.io` });

  socket.on('connect', () => console.log('[live] conectado:', socket.id));

  socket.on('disconnect', () => console.log('[live] desconectado'));

  socket.on('event', (e) => {

    const tbody = document.querySelector('#panel .table tbody')

               || document.getElementById('tbl-events')

               || document.querySelector('#panel tbody');

    if (!tbody) return;

    const tr = document.createElement('tr');

    tr.innerHTML = `

      <td>${new Date(e.ts || Date.now()).toLocaleString()}</td>

      <td>${e.origin || '-'}</td>

      <td>${e.event || e.command || '-'}</td>

      <td><span class="status"><span class="dot ok"></span>OK</span></td>`;

    tbody.prepend(tr);

    trimTableBodyRows(tbody, LOG_MAX_ROWS);

  });

  socket.on('metrics', (payload = {}) => {

    try { renderStatusMetrics(payload.metrics || payload); } catch (err) { console.error('[live] metrics handler failed', err); }

  });

  socket.on('member_role', (payload = {}) => {

    try {

      const userId = payload.userId || payload.discordId;

      if (!userId) return;

      if (!window.CURRENT_USER || window.CURRENT_USER.id !== userId) {

        if (!window.CURRENT_USER) window.CURRENT_USER = { id: userId };

        else return;

      }

      const name = payload.roleName ?? payload.name ?? null;

      const colorRaw = typeof payload.roleColor === 'number' ? payload.roleColor : (typeof payload.color === 'number' ? payload.color : null);

      if (name != null) window.CURRENT_USER.discordRoleName = name;

      if (colorRaw != null) window.CURRENT_USER.discordRoleColor = colorRaw;

      populateProfileModal(window.CURRENT_USER);

    } catch (err) {

      console.error('[live] member_role handler failed', err);

    }

  });

}



// ================================== Arranque Frontend ==================================

document.addEventListener('DOMContentLoaded', async ()=>{

  // Tema inicial y UI del selector

  applyTheme('dark', { persist:false });

  initThemeUI();

  const basePath = (new URL(API_BASE)).pathname.replace(/\/+$/, '');

  // Debug banner eliminado



  // Probar si el backend responde en API_BASE; si no, caer a puerto 3001

  try {

    const res = await fetch(API_BASE + '/api/health', { credentials: 'include' });

    if (!res.ok) throw new Error('health not ok');

  } catch {

    API_BASE = `${window.location.protocol}//${window.location.hostname}:3001`;

    // fallback silencioso

  }



  // Reescribe hrefs de navegación para mantener la subruta (ej. /panel/#logs)

  initPanelLogin();
  handleDiscordLinkParams();

  document.querySelectorAll('.nav a[data-route]')?.forEach(a => {

    const route = a.getAttribute('data-route') || '';

    a.setAttribute('href', `${basePath}/#${route || 'panel'}`);

  });



  // Botón para vincular Discord tras iniciar sesión

  const connectBtnSidebar = document.getElementById('btn-connect-discord');

  if (connectBtnSidebar) {

    const connectUrl = `${API_BASE}/auth/discord/login`;

    connectBtnSidebar.setAttribute('href', connectUrl);

    connectBtnSidebar.addEventListener('click', (e)=>{

      e.preventDefault();

      window.location.assign(connectUrl);

    });

  }

  // Toggle menú usuario

  const pill = document.getElementById('user-pill');

  const menu = document.getElementById('user-menu');

  pill?.addEventListener('click', (e)=>{

    e.preventDefault();

    if (!menu) return;

    const visible = menu.style.display === 'block';

    menu.style.display = visible ? 'none' : 'block';

  });

  document.addEventListener('click', (e)=>{

    if (!menu || !pill) return;

    if (!pill.contains(e.target)) menu.style.display='none';

  });

  // Logout

  const logoutMenu = document.getElementById('menu-logout');

  const profileMenu = document.getElementById('menu-profile');

  if (logoutMenu) {

    logoutMenu.addEventListener('click', async (e)=>{

      e.preventDefault();

      try { await fetchJSON('/api/auth/logout', { method: 'POST' }); } catch {}

      const basePath = (new URL(API_BASE)).pathname.replace(/\/+$/, '');

      window.location.assign(basePath || '/');

    });

  }

  if (profileMenu) {

    profileMenu.addEventListener('click', async (e)=>{

      e.preventDefault();

      const dlg = document.getElementById('modal-profile');

      if (!dlg) return;

      try {

        const { user } = await fetchJSON('/api/me');

        window.CURRENT_USER = user;

      } catch {}

      populateProfileModal(window.CURRENT_USER);

      try { dlg.showModal(); } catch { dlg.open = true; }

    });

  }

  // Admin modal open

  const adminBtn = document.getElementById('btn-admin');

  const adminDlg = document.getElementById('modal-admin');

  adminBtn?.addEventListener('click', (e)=>{

    e.preventDefault();

    initAdminModal();

    switchAdminSection('users');

    const codeInput = document.querySelector('#admin-user-create input[name="accountCode"]');

    if (codeInput && !codeInput.value) codeInput.value = generateFrontendAccountCode();

    loadAdminAccounts().catch(()=> setAdminMessage('admin-user-message', 'No se pudieron cargar usuarios', 'error'));

    loadAdminRoles().catch(()=> setAdminMessage('admin-roles-message', 'No se pudieron cargar permisos', 'error'));

    try { adminDlg?.showModal(); } catch { if (adminDlg) adminDlg.open = true; }

  });

  document.getElementById('btn-ranks-refresh')?.addEventListener('click', async (ev)=>{

    const btn = ev.currentTarget instanceof HTMLElement ? ev.currentTarget : null;

    btn?.setAttribute('disabled','');

    try {

      await refreshDiscordRanks({ successMessage: 'Datos actualizados' });

    } catch {} finally {

      btn?.removeAttribute('disabled');

    }

  });

  document.getElementById('btn-rank-save')?.addEventListener('click', async (ev)=>{

    const btn = ev.currentTarget instanceof HTMLElement ? ev.currentTarget : null;

    const sel = document.getElementById('rank-user-select');

    const uid = sel?.value?.trim();

    const rank = document.getElementById('rank-select')?.value;

    if (!uid || !rank) return toast('Faltan datos', 'error');

    btn?.setAttribute('disabled','');

    try {

      await fetch(API_BASE + '/api/ranks/' + encodeURIComponent(uid), {

        method: 'POST',

        credentials: 'include',

        headers: { 'Content-Type': 'application/json' },

        body: JSON.stringify({ rank })

      });

      toast('Rango actualizado');

      await refreshDiscordRanks({ successMessage: 'Rango actualizado', selected: uid });

    } catch {

      toast('No se pudo actualizar', 'error');

    } finally {

      btn?.removeAttribute('disabled');

    }

  });

  document.getElementById('btn-restart')?.addEventListener('click', async ()=>{

    const code = document.getElementById('sudo-code')?.value?.trim();

    if (!code) return toast('Introduce SUDO code', 'error');

    try {

      const res = await fetch(API_BASE + '/api/actions/restart', {

        method: 'POST', credentials: 'include',

        headers: { 'Content-Type': 'application/json', 'x-sudo-code': code, 'x-sudo-ok': '1' },

      }).then(r=>r.json());

      if (res?.ok) toast('Solicitud de reinicio enviada'); else toast('No se pudo reiniciar', 'error');

    } catch { toast('Error al reiniciar', 'error'); }

  });

  document.getElementById('btn-cleanup-logs')?.addEventListener('click', async ()=>{

    const code = document.getElementById('sudo-code')?.value?.trim();

    const days = parseInt(document.getElementById('cleanup-days')?.value || '30', 10);

    if (!code) return toast('Introduce SUDO code', 'error');

    try {

      const res = await fetch(API_BASE + '/api/actions/cleanup-logs', {

        method: 'POST', credentials: 'include',

        headers: { 'Content-Type': 'application/json', 'x-sudo-code': code, 'x-sudo-ok': '1' },

        body: JSON.stringify({ olderThanDays: days })

      }).then(r=>r.json());

      if (res?.ok) { toast(`Logs limpiados: ${res.deleted||0}`); await loadLogs().catch(()=>{}); }

      else toast('No se pudo limpiar', 'error');

    } catch { toast('Error al limpiar', 'error'); }

  });

  // (Eliminado) opción de copiar ID desde el menú de la pill

  const me = await refreshAuthUI();

  await onAuthenticated(me);

});



// ===================== Perfil: rellenar modal + cierres =====================

function populateProfileModal(u){

  if (!u) u = window.CURRENT_USER || {};

  const avatarEl = document.getElementById('profile-avatar');

  const userEl = document.getElementById('profile-username');

  const didEl = document.getElementById('profile-discord-id');

  const pidEl = document.getElementById('profile-panel-id');

  const prankEl = document.getElementById('profile-panel-rank');

  const drankEl = document.getElementById('profile-discord-rank');

  const loginAtEl = document.getElementById('profile-login-at');

  const loginAgoEl = document.getElementById('profile-login-ago');

  if (avatarEl) {

    const avatarUrl = u?.avatarUrl || (u?.avatar ? `https://cdn.discordapp.com/avatars/${u.id}/${u.avatar}.png?size=128` : 'https://cdn.discordapp.com/embed/avatars/0.png');

    avatarEl.src = avatarUrl;

    avatarEl.alt = u?.username || 'user';

  }

  if (userEl) userEl.textContent = u?.username || '-';

  if (didEl) didEl.textContent = u?.discordId || '-';

  const accountIdentifier = (u?.panelId != null) ? String(u.panelId) : (u?.accountId != null ? String(u.accountId) : '-');

  if (pidEl) pidEl.textContent = accountIdentifier;

  if (prankEl) prankEl.textContent = (u?.rank || '').toString() || '-';

  if (drankEl) {

    const name = u?.discordId ? (u?.discordRoleName || 'N/D') : 'Sin vincular';

    drankEl.textContent = name;

    try {

      const c = u?.discordRoleColor;

      if (typeof c === 'number' && c > 0) {

        const hex = '#' + (c >>> 0).toString(16).padStart(6, '0');

        drankEl.style.color = hex;

      } else {

        drankEl.style.color = 'inherit';

      }

    } catch { drankEl.style.color = 'inherit'; }

  }

  const loginTimestamp =

    typeof u?.sessionStartedAt === 'number' ? Number(u.sessionStartedAt) :

    (typeof u?.lastLoginAt === 'number' ? Number(u.lastLoginAt) :

    (typeof u?.updatedAt === 'number' ? Number(u.updatedAt) :

    (typeof u?.iat === 'number' ? Number(u.iat) * 1000 : null)));

  if (loginTimestamp && Number.isFinite(loginTimestamp)) {

    const loginDate = new Date(loginTimestamp);

    if (!Number.isNaN(loginDate.getTime())) {

      if (loginAtEl) loginAtEl.textContent = loginDate.toLocaleString();

      if (loginAgoEl) loginAgoEl.textContent = timeSince(loginDate);

    } else {

      if (loginAtEl) loginAtEl.textContent = '-';

      if (loginAgoEl) loginAgoEl.textContent = '-';

    }

  } else {

    if (loginAtEl) loginAtEl.textContent = '-';

    if (loginAgoEl) loginAgoEl.textContent = '-';

  }

}



function timeSince(date){

  const seconds = Math.floor((Date.now() - date.getTime()) / 1000);

  const units = [

    ['año', 365*24*3600],

    ['mes', 30*24*3600],

    ['día', 24*3600],

    ['hora', 3600],

    ['minuto', 60],

    ['segundo', 1],

  ];

  for (const [name, s] of units){

    const v = Math.floor(seconds / s);

    if (v >= 1) return `${v} ${name}${v>1?'s':''}`;

  }

  return 'instantes';

}



// Cerrar modal perfil

document.getElementById('profile-close')?.addEventListener('click', ()=>{

  const dlg = document.getElementById('modal-profile');

  try { dlg?.close(); } catch { if (dlg) dlg.open = false; }

});

document.getElementById('profile-close-2')?.addEventListener('click', ()=>{

  const dlg = document.getElementById('modal-profile');

  try { dlg?.close(); } catch { if (dlg) dlg.open = false; }

});



// Copiar Panel ID desde el modal de perfil

document.getElementById('profile-copy-id')?.addEventListener('click', async ()=>{

  try {

    const id = (window.CURRENT_USER?.panelId != null) ? String(window.CURRENT_USER.panelId) : '';

    if (!id) return toast('Sin Panel ID', 'error');

    await navigator.clipboard.writeText(id);

    toast('Panel ID copiado');

  } catch {

    toast('No se pudo copiar', 'error');

  }

});



// ================== Tickets: botón de sincronización manual ==================

document.getElementById('btnTicketsSync')?.addEventListener('click', async ()=>{

  try {

    await fetchJSON('/api/tickets/sync', { method: 'POST' });

    toast('Sincronización de tickets solicitada');

    await loadTickets();

  } catch {

    toast('No se pudo sincronizar tickets', 'error');

  }

});



// ================== Admin: cargar lista de rangos ==================

async function loadRanks(){

  const rows = await fetchJSON('/api/ranks');

  const list = Array.isArray(rows) ? rows.slice() : [];

  const tbody = document.getElementById('tbl-ranks');

  if (!tbody) return list;

  tbody.innerHTML = '';

  if (!list.length) {

    const tr = document.createElement('tr');

    tr.innerHTML = '<td class="empty-row" colspan="3">Sin rangos registrados</td>';

    tbody.appendChild(tr);

    return list;

  }

  list.sort((a, b) => {

    const rankA = (a.rank || '').toString().toLowerCase();

    const rankB = (b.rank || '').toString().toLowerCase();

    if (rankA === rankB) {

      return (a.userId || '').toString().localeCompare((b.userId || '').toString());

    }

    return rankA.localeCompare(rankB);

  });

  list.forEach(r=>{

    const tr = document.createElement('tr');

    tr.innerHTML = `<td><code>${r.userId}</code></td><td>${r.guildId || '-'}</td><td>${r.rank}</td>`;

    tbody.appendChild(tr);

  });

  return list;

}



async function loadDiscordUsers(opts = {}){

  const rows = await fetchJSON('/api/ranks/users');

  const list = Array.isArray(rows) ? rows.slice() : [];

  const sel = document.getElementById('rank-user-select');

  if (!sel) return list;

  const selected = typeof opts.selected === 'string' ? opts.selected : '';

  sel.innerHTML = '';

  if (!list.length) {

    const opt = document.createElement('option');

    opt.value = '';

    opt.textContent = 'Sin usuarios disponibles';

    opt.disabled = true;

    opt.selected = true;

    sel.appendChild(opt);

    return list;

  }

  list.sort((a, b) => (a.username || '').localeCompare(b.username || '', 'es', { sensitivity: 'base' }));

  const placeholder = document.createElement('option');

  placeholder.value = '';

  placeholder.textContent = 'Selecciona un usuario';

  placeholder.disabled = true;

  placeholder.selected = !selected;

  sel.appendChild(placeholder);

  list.forEach(u => {

    const opt = document.createElement('option');

    const labelName = u.username || 'Usuario';

    opt.value = u.discordId || '';

    opt.textContent = `${labelName} (${u.panelId ?? '-'})`;

    sel.appendChild(opt);

  });

  if (selected && list.some(u => u.discordId === selected)) {

    sel.value = selected;

  }

  return list;

}



async function refreshDiscordRanks(opts = {}){

  const messageId = 'admin-ranks-message';

  const userSelect = document.getElementById('rank-user-select');

  const current = typeof opts.selected === 'string' ? opts.selected : (userSelect?.value || '');

  setAdminMessage(messageId, 'Cargando datos de Discord...', 'info');

  try {

    const [users, ranks] = await Promise.all([

      loadDiscordUsers({ selected: current }),

      loadRanks()

    ]);

    let message = '';

    let kind = 'info';

    if (!users.length && !ranks.length) {

      message = 'No hay usuarios vinculados ni rangos configurados todavia. Verifica que el bot tenga habilitado el intent GUILD_MEMBERS y que la sincronizacion hacia la base de datos este en marcha.';

    } else if (!users.length) {

      message = 'No hay usuarios de Discord vinculados al panel. Verifica que el bot tenga el intent GUILD_MEMBERS y que la sincronizacion hacia panel_users este en marcha.';

    } else if (!ranks.length) {

      message = 'No hay rangos configurados todavía.';

    } else if (opts.successMessage) {

      message = opts.successMessage;

      kind = 'success';

    }

    if (message) setAdminMessage(messageId, message, kind);

    else setAdminMessage(messageId);

    if (userSelect && current && users.some(u => u.discordId === current)) {

      userSelect.value = current;

    }

    return { users, ranks };

  } catch (err) {

    console.error('[admin] refresh discord ranks error', err);

    setAdminMessage(messageId, 'No se pudieron cargar los datos de Discord', 'error');

    throw err;

  }

}



