import fetch from 'node-fetch';
import { signJwt, verifyJwt } from './jwt.js';
import {
  getPanelAccountById,
  getPanelAccountByDiscordId,
  getOrCreatePanelId,
  updatePanelUserInfo,
  linkPanelAccountDiscord,
} from '../utils/db.js';

const OAUTH_AUTHORIZE = 'https://discord.com/api/oauth2/authorize';
const OAUTH_TOKEN = 'https://discord.com/api/oauth2/token';
const API_USERS_ME = 'https://discord.com/api/users/@me';
// const API_USERS_GUILDS = 'https://discord.com/api/users/@me/guilds';

// Scopes requested from Discord
const scopes = ['identify', 'guilds'];

/**
 * Redirects the user to Discord's OAuth2 authorization screen.
 */
export function discordLogin(req, res) {
  const params = new URLSearchParams({
    client_id: process.env.DISCORD_CLIENT_ID,
    redirect_uri: process.env.DISCORD_REDIRECT_URI,
    response_type: 'code',
    scope: scopes.join(' '),
  });
  res.redirect(`${OAUTH_AUTHORIZE}?${params.toString()}`);
}

function cookieOptions(req) {
  const sameSite = process.env.COOKIE_SAMESITE || 'lax';
  const secure = (process.env.COOKIE_SECURE || '').toLowerCase() === 'true' || req.secure;
  const path = process.env.PUBLIC_BASE_PATH
    ? `/${String(process.env.PUBLIC_BASE_PATH).replace(/\/+$/, '').replace(/^\/+/, '')}`
    : '/';
  return { httpOnly: true, sameSite, secure, maxAge: 1000 * 60 * 60 * 24 * 7, path };
}

function buildRedirect(baseUrl, status) {
  if (!status) return baseUrl;
  const hashIndex = baseUrl.indexOf('#');
  const hash = hashIndex >= 0 ? baseUrl.slice(hashIndex) : '';
  const clean = hashIndex >= 0 ? baseUrl.slice(0, hashIndex) : baseUrl;
  const separator = clean.includes('?') ? '&' : '?';
  return `${clean}${separator}discord=${encodeURIComponent(status)}${hash}`;
}

function buildRedirectTarget() {
  const base = (process.env.PUBLIC_BASE_PATH || '').replace(/\/+$/, '').replace(/^\/+/, '');
  return process.env.FRONTEND_URL || (base ? `/${base}/` : '/');
}

function signPanelSession(account) {
  return signJwt({
    accountId: account.id,
    accountCode: account.accountCode || null,
    username: account.username,
    role: account.role,
    avatarUrl: account.avatarUrl || null,
    discordId: account.discordId || null,
    type: 'panel',
  });
}

/**
 * Handles the OAuth2 callback from Discord. When a panel session exists the Discord
 * account is linked to that panel account and the session cookie is refreshed.
 */
export async function discordCallback(req, res) {
  const startedAt = Date.now();
  const redirectBase = buildRedirectTarget();
  const options = cookieOptions(req);
  const clearOptions = { ...options };
  delete clearOptions.maxAge;
  try {
    const code = req.query.code;
    if (!code) {
      console.warn('[oauth] callback without code');
      return res.redirect(buildRedirect(redirectBase, 'error'));
    }

    const existingToken = req.cookies?.auth;
    const existingSession = existingToken ? verifyJwt(existingToken) : null;
    if (!existingSession?.accountId) {
      console.warn('[oauth] callback without panel session, ignoring link');
      res.clearCookie('auth', clearOptions);
      return res.redirect(buildRedirect(redirectBase, 'no-session'));
    }

    // Exchange code for access token (with timeout)
    const controller = typeof AbortController !== 'undefined' ? new AbortController() : null;
    const timer = setTimeout(() => controller?.abort(), 12_000);

    const body = new URLSearchParams({
      client_id: process.env.DISCORD_CLIENT_ID,
      client_secret: process.env.DISCORD_CLIENT_SECRET,
      grant_type: 'authorization_code',
      code,
      redirect_uri: process.env.DISCORD_REDIRECT_URI,
      scope: scopes.join(' '),
    });

    const tokenRes = await fetch(OAUTH_TOKEN, {
      method: 'POST',
      headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
      body,
      signal: controller?.signal,
    })
      .then(async (r) => { let j=null; try{ j=await r.json(); } catch(e){ console.error('[oauth] token parse error:', e);} return j; })
      .catch((e)=>{ console.error('[oauth] token fetch error:', e?.message||e); throw e; })
      .finally(()=> clearTimeout(timer));

    if (!tokenRes?.access_token) {
      console.warn('[oauth] missing access token from Discord');
      return res.redirect(buildRedirect(redirectBase, 'error'));
    }

    // Fetch user info
    const me = await fetch(API_USERS_ME, {
      headers: { Authorization: `Bearer ${tokenRes.access_token}` },
    }).then((r) => r.json()).catch((e)=>{ console.error('[oauth] /users/@me error:', e); throw e; });
    if (!me?.id) {
      console.warn('[oauth] missing user info');
      return res.redirect(buildRedirect(redirectBase, 'error'));
    }

    const account = await getPanelAccountById(existingSession.accountId);
    if (!account) {
      console.warn('[oauth] session account not found');
      res.clearCookie('auth', clearOptions);
      return res.redirect(buildRedirect(redirectBase, 'no-session'));
    }

    if (account.discordId && account.discordId !== me.id) {
      console.info('[oauth] overwriting previous discord link for account %s', account.id);
    }

    try {
      const alreadyLinked = await getPanelAccountByDiscordId(me.id);
      if (alreadyLinked && alreadyLinked.id !== account.id) {
        console.warn('[oauth] discord id already linked to account %s', alreadyLinked.id);
        return res.redirect(buildRedirect(redirectBase, 'conflict'));
      }
      await linkPanelAccountDiscord(account.id, me.id);
    } catch (err) {
      if (err?.code === 'ER_DUP_ENTRY') {
        console.warn('[oauth] discord id already linked elsewhere');
        return res.redirect(buildRedirect(redirectBase, 'conflict'));
      }
      throw err;
    }

    // Persist basic Discord profile info for downstream features
    try { await getOrCreatePanelId(me.id); } catch {}
    try {
      const displayName = me.global_name || me.username || '';
      await updatePanelUserInfo(me.id, { username: displayName, avatar: me.avatar || null });
    } catch {}

    const updated = await getPanelAccountById(account.id);
    const token = signPanelSession(updated);
    res.cookie('auth', token, options);
    const redirectTo = buildRedirect(redirectBase, 'linked');
    console.log(`[oauth] linked discord -> account ${account.id} in ${Date.now() - startedAt}ms`);
    return res.redirect(redirectTo);
  } catch (e) {
    console.error('[oauth] callback fatal:', e?.stack || e?.message || e);
    return res.redirect(buildRedirect(redirectBase, 'error'));
  }
}
