import 'dotenv/config';
import express from 'express';
import cookieParser from 'cookie-parser';
import { createServer } from 'http';
import { Server } from 'socket.io';
import cors from 'cors';
import path from 'path';
import { fileURLToPath } from 'url';

import { discordLogin, discordCallback } from './auth/discord-oauth.js';
import meRouter from './routes/me.js';
import kpisRouter from './routes/kpis.js';
import logsRouter from './routes/logs.js';
import ticketsRouter from './routes/tickets.js';
import actionsRouter from './routes/actions.js';
import { setupSocket } from './live/socket.js';
// Import database initialization
import { initialize as initDb } from './utils/db.js';
import { corsConfig } from './utils/cors.js';
import ranksRouter from './routes/ranks.js';
import statusRouter from './routes/status.js';
import authPanelRouter from './routes/auth-panel.js';
import adminRouter from './routes/admin.js';

const app = express();
// Optional base path so app works under /panel without Apache rewrite
const BASE_PATH = (process.env.PUBLIC_BASE_PATH || '').replace(/\/+$/, '').replace(/^\/+/, '');
const base = BASE_PATH ? `/${BASE_PATH}` : '';
// Detrás de Apache/Nginx, necesario para cookies y req.secure
app.set('trust proxy', 1);
// Create HTTP server and bind Socket.IO to it
const server = createServer(app);
const corsOpt = process.env.CORS_ORIGIN ? { origin: process.env.CORS_ORIGIN, credentials: true } : undefined;
const ioRoot = new Server(server, { path: '/socket.io', cors: corsOpt });
const ioBase = base ? new Server(server, { path: `${base}/socket.io`, cors: corsOpt }) : null;

// Middlewares
if (process.env.CORS_ORIGIN) {
  app.use(cors(corsConfig));
}
app.use(express.json());
app.use(cookieParser());

// OAuth routes
app.get('/auth/discord/login', discordLogin);
app.get('/auth/discord/callback', discordCallback);
// Logout: limpia la cookie 'auth'
app.get('/auth/logout', (req, res) => {
  const basePath = (process.env.PUBLIC_BASE_PATH || '').replace(/\/+$/, '').replace(/^\/+/, '');
  const path = basePath ? `/${basePath}` : '/';
  const sameSite = process.env.COOKIE_SAMESITE || 'lax';
  const secure = (process.env.COOKIE_SECURE || '').toLowerCase() === 'true' || req.secure;
  res.clearCookie('auth', { path, httpOnly: true, sameSite, secure });
  res.json({ ok: true });
});
if (base) {
  app.get(`${base}/auth/discord/login`, discordLogin);
  app.get(`${base}/auth/discord/callback`, discordCallback);
  app.get(`${base}/auth/logout`, (req, res) => {
    const sameSite = process.env.COOKIE_SAMESITE || 'lax';
    const secure = (process.env.COOKIE_SECURE || '').toLowerCase() === 'true' || req.secure;
    res.clearCookie('auth', { path: base || '/', httpOnly: true, sameSite, secure });
    res.json({ ok: true });
  });
}

// API routes (protected)
app.use('/api/me', meRouter);
app.use('/api/kpis', kpisRouter);
app.use('/api/logs', logsRouter);
app.use('/api/tickets', ticketsRouter);
app.use('/api/actions', actionsRouter);
app.use('/api/ranks', ranksRouter);
app.use('/api/status', statusRouter);
app.use('/api/auth', authPanelRouter);
app.use('/api/admin', adminRouter);
// Health endpoint for diagnostics
app.get('/api/health', (_req, res) => {
  res.json({ ok: true, env: process.env.NODE_ENV || 'development', ts: Date.now() });
});
if (base) {
  app.use(`${base}/api/me`, meRouter);
  app.use(`${base}/api/kpis`, kpisRouter);
  app.use(`${base}/api/logs`, logsRouter);
  app.use(`${base}/api/tickets`, ticketsRouter);
  app.use(`${base}/api/actions`, actionsRouter);
  app.use(`${base}/api/ranks`, ranksRouter);
  app.use(`${base}/api/status`, statusRouter);
  app.use(`${base}/api/auth`, authPanelRouter);
  app.use(`${base}/api/admin`, adminRouter);
  app.get(`${base}/api/health`, (_req, res) => {
    res.json({ ok: true, env: process.env.NODE_ENV || 'development', ts: Date.now(), base });
  });
}

// Static frontend (serve index.html and assets from project root)
const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);
const webRoot = path.join(__dirname, '..');
app.use(express.static(webRoot));
app.get('/', (_req, res) => {
  res.sendFile(path.join(webRoot, 'index.html'));
});
if (base) {
  app.use(base || '/', express.static(webRoot));
  app.get(base || '/', (_req, res) => {
    res.sendFile(path.join(webRoot, 'index.html'));
  });
}

// Initialize database then start server and websockets
(async () => {
  try {
    await initDb();
    // Setup WebSockets after DB is ready
    setupSocket(ioRoot);
    if (ioBase) setupSocket(ioBase);
    const PORT = process.env.PORT || 3001;
    server.listen(PORT, () => {
      console.log(`[panel] listening on http://localhost:${PORT}`);
    });
  } catch (err) {
    console.error('[panel] failed to initialize DB:', err);
    process.exit(1);
  }
})();
