/**
 * Configures Socket.IO namespaces for UI and bot connections. The UI connects
 * to `/live` without authentication, while the bot connects to `/bot`
 * authenticated by BOT_PANEL_TOKEN. Incoming events from the bot are
 * forwarded to all connected UI clients. Bot actions triggered via REST
 * endpoints are sent back to the bot using a global helper.
 *
 * @param {import('socket.io').Server} io
 */
import { query, updatePanelUserDiscordRole } from '../utils/db.js';

// Última métrica recibida del bot
let latestStatus = { ts: 0, data: null };
export function getStatusSnapshot() {
  return latestStatus;
}

export function setupSocket(io) {
  // Public namespace for dashboard clients
  const ui = io.of('/live');
  ui.on('connection', (socket) => {
    console.log('[socket] UI connected', socket.id);
  });

  // Private namespace for the Discord bot
  const bot = io.of('/bot');
  // Authenticate bot connections using the token
  bot.use((socket, next) => {
    const token = socket.handshake.auth?.token;
    if (token !== process.env.BOT_PANEL_TOKEN) return next(new Error('unauthorized'));
    next();
  });

  bot.on('connection', (socket) => {
    console.log('[socket] BOT connected', socket.id);

    // Expose a global helper to send actions to the bot
    global.emitToBot = (event, payload) => socket.emit(event, payload);

    // Métricas periódicas del bot
    socket.on('metrics', (payload) => {
      latestStatus = { ts: Date.now(), data: payload };
      try { ui.emit('metrics', payload); } catch (err) { console.error('[socket] metrics broadcast failed:', err); }
    });

    // Relay events from bot to all UI clients and persist them in DB
    socket.on('event', async (payload) => {
      try {
        const guildId = payload.guildId || null;
        const userId = payload.userId || null;
        const eventType = payload.origin || payload.eventType || 'unknown';
        const details = JSON.stringify(payload);
        const timestamp = Date.now();
        await query(
          'INSERT INTO detailed_logs (guildId, userId, eventType, details, timestamp) VALUES (?, ?, ?, ?, ?)',
          [guildId, userId, eventType, details, timestamp]
        );

        // Ticket sync: upsert current state into tickets table
        if (payload?.type === 'ticket_open') {
          const createdAt = Number(payload.createdAt || timestamp);
          const staffId = payload.staffId || null;
          const category = payload.category || null;
          await query(
            `INSERT INTO tickets (channelId, guildId, openerId, staffId, category, status, createdAt, updatedAt)
             VALUES (?, ?, ?, ?, ?, 'open', ?, ?)
             ON DUPLICATE KEY UPDATE staffId = VALUES(staffId), category = VALUES(category), status='open', updatedAt = VALUES(updatedAt)`,
            [payload.channelId, guildId, payload.openerId || null, staffId, category, createdAt, timestamp]
          );
        } else if (payload?.type === 'ticket_claim') {
          await query(
            `UPDATE tickets SET staffId = ?, status = 'open', updatedAt = ? WHERE channelId = ?`,
            [payload.staffId || null, timestamp, payload.channelId]
          );
        } else if (payload?.type === 'ticket_close') {
          await query(
            `UPDATE tickets SET status = 'closed', updatedAt = ? WHERE channelId = ?`,
            [timestamp, payload.channelId]
          );
        }
      } catch (err) {
        console.error('[socket] Error persisting event:', err);
      }
      // Forward to UI clients
      ui.emit('event', payload);
    });

    // Optional: bot can push member's top role info for UI profile modal
    socket.on('member_role', async (payload = {}) => {
      try {
        const userId = payload.userId || payload.discordId;
        if (!userId) return;
        const name = payload.name || payload.roleName || null;
        const color = typeof payload.color === 'number' ? payload.color : (typeof payload.roleColor === 'number' ? payload.roleColor : null);
        await updatePanelUserDiscordRole(userId, { roleName: name, roleColor: color });
        const sanitized = { userId, discordId: userId, roleName: name, roleColor: color };
        ui.emit('member_role', sanitized);
      } catch (e) {
        console.error('[socket] member_role update failed:', e);
      }
    });
  });
}
