import { verifyJwt } from '../auth/jwt.js';
import { getPanelAccountById } from '../utils/db.js';

/**
 * Middleware to ensure a request has a valid JWT. It looks for a cookie named
 * `auth` or a Bearer token in the Authorization header, verifies it, and
 * attaches the user payload to `req.user`. If verification fails, it returns
 * a 401 Unauthorized response.
 */
export async function requireAuth(req, res, next) {
  const token = req.cookies?.auth || (req.headers.authorization?.replace('Bearer ', ''));
  const payload = token ? verifyJwt(token) : null;
  if (!payload?.accountId) {
    return res.status(401).json({ error: 'unauthorized' });
  }

  try {
    const account = await getPanelAccountById(payload.accountId);
    if (!account) return res.status(401).json({ error: 'unauthorized' });
    req.panelAccount = {
      id: account.id,
      accountCode: account.accountCode || null,
      username: account.username,
      role: account.role,
      avatarUrl: account.avatarUrl || null,
      discordId: account.discordId || null,
      createdAt: account.createdAt || null,
      updatedAt: account.updatedAt || null,
    };
    req.user = {
      id: account.id,
      accountId: account.id,
      accountCode: account.accountCode || null,
      username: account.username,
      role: account.role,
      rank: account.role,
      avatar: account.avatarUrl || null,
      avatarUrl: account.avatarUrl || null,
      discordId: account.discordId || null,
      type: 'panel',
    };
    next();
  } catch (err) {
    console.error('[auth] session lookup failed', err);
    res.status(500).json({ error: 'session_error' });
  }
}
