import { Router } from 'express';
import { requireAuth } from '../middlewares/requireAuth.js';
import { requireAdmin } from '../middlewares/requireRole.js';
import { requireSudo } from '../security/sudo-guard.js';
import { query } from '../utils/db.js';

const r = Router();

// Endpoint to request a bot restart via WebSocket; only admins with sudo
r.post('/restart', requireAuth, requireAdmin, requireSudo, async (req, res) => {
  // Emit a restart action to the connected bot via global helper
  const actor = req.user?.discordId || req.user?.accountCode || String(req.user?.id || '');
  const ok = global.emitToBot?.('action:restart', {
    by: actor,
    accountId: req.user?.accountId || req.user?.id || null,
    accountCode: req.user?.accountCode || null,
    discordId: req.user?.discordId || null,
    username: req.user?.username || null,
  });
  res.json({ ok: !!ok });
});

export default r;

// Maintenance: cleanup logs older than N days
r.post('/cleanup-logs', requireAuth, requireAdmin, requireSudo, async (req, res) => {
  try {
    const olderThanDays = Math.max(1, Math.min(365, parseInt(req.body?.olderThanDays ?? '30', 10)));
    const cutoff = Date.now() - olderThanDays * 24 * 60 * 60 * 1000;
    const result = await query('DELETE FROM detailed_logs WHERE timestamp < ?', [cutoff]);
    // mysql2 returns an object with affectedRows when using pool.query; but here query() returns rows array for SELECTs.
    // For DELETE, mysql2 returns [result] where result.affectedRows is available; our query() returns first item; adjust:
    const deleted = result?.affectedRows ?? 0;
    res.json({ ok: true, deleted, olderThanDays });
  } catch (e) {
    console.error('[actions] cleanup-logs error:', e);
    res.status(500).json({ ok: false, error: 'cleanup_failed' });
  }
});
