import { Router } from 'express';
import { requireAuth } from '../middlewares/requireAuth.js';
import {
  listPanelAccounts,
  createPanelAccount,
  updatePanelAccount,
  setPanelAccountPassword,
  listRolePermissions,
  setRolePermissions,
} from '../utils/db.js';
import { hashPassword } from '../security/password.js';
import { DEFAULT_ROLE_PERMISSIONS, resolveRolePermissions } from '../utils/permissions.js';

const router = Router();

function ensureAdmin(req, res, next) {
  const role = String(req.user?.role || '').toLowerCase();
  if (!['admin', 'superadmin'].includes(role)) {
    return res.status(403).json({ error: 'forbidden' });
  }
  next();
}

function sanitizeAccount(account) {
  if (!account) return null;
  return {
    id: account.id,
    accountCode: account.accountCode || null,
    username: account.username,
    role: account.role,
    avatarUrl: account.avatarUrl || null,
    discordId: account.discordId || null,
    createdAt: account.createdAt || null,
    updatedAt: account.updatedAt || null,
  };
}

router.use(requireAuth, ensureAdmin);

router.get('/accounts', async (_req, res) => {
  const rows = await listPanelAccounts();
  res.json({ accounts: rows.map(sanitizeAccount) });
});

router.post('/accounts', async (req, res) => {
  try {
    const { accountCode, username, password, role, avatarUrl = null, discordId = null } = req.body || {};
    if (typeof username !== 'string' || !username.trim()) return res.status(400).json({ error: 'invalid_username' });
    if (typeof role !== 'string' || !role.trim()) return res.status(400).json({ error: 'invalid_role' });
    if (typeof password !== 'string' || password.length < 4) return res.status(400).json({ error: 'invalid_password' });
    const code = (typeof accountCode === 'string' && accountCode.trim()) ? accountCode.trim() : generateAccountCode();
    const passwordHash = await hashPassword(password);
    const { id } = await createPanelAccount({ accountCode: code, username: username.trim(), passwordHash, role: role.trim(), avatarUrl, discordId });
    const accounts = await listPanelAccounts();
    res.status(201).json({ ok: true, id, accounts: accounts.map(sanitizeAccount) });
  } catch (err) {
    console.error('[admin] create account error', err);
    res.status(500).json({ error: 'create_failed' });
  }
});

router.put('/accounts/:id', async (req, res) => {
  try {
    const id = Number(req.params.id);
    if (!Number.isFinite(id)) return res.status(400).json({ error: 'invalid_id' });
    const fields = {};
    const allowed = ['accountCode', 'username', 'role', 'avatarUrl', 'discordId'];
    for (const key of allowed) {
      if (Object.prototype.hasOwnProperty.call(req.body || {}, key)) {
        const value = req.body[key];
        if (value === null || value === '') {
          fields[key] = null;
        } else if (typeof value === 'string') {
          fields[key] = value.trim();
        }
      }
    }
    if (!Object.keys(fields).length) return res.status(400).json({ error: 'no_changes' });
    const ok = await updatePanelAccount(id, fields);
    if (!ok) return res.status(404).json({ error: 'not_found' });
    const accounts = await listPanelAccounts();
    res.json({ ok: true, accounts: accounts.map(sanitizeAccount) });
  } catch (err) {
    console.error('[admin] update account error', err);
    res.status(500).json({ error: 'update_failed' });
  }
});

router.post('/accounts/:id/password', async (req, res) => {
  try {
    const id = Number(req.params.id);
    if (!Number.isFinite(id)) return res.status(400).json({ error: 'invalid_id' });
    const { password } = req.body || {};
    if (typeof password !== 'string' || password.length < 4) return res.status(400).json({ error: 'invalid_password' });
    const passwordHash = await hashPassword(password);
    const ok = await setPanelAccountPassword(id, passwordHash);
    if (!ok) return res.status(404).json({ error: 'not_found' });
    res.json({ ok: true });
  } catch (err) {
    console.error('[admin] reset password error', err);
    res.status(500).json({ error: 'password_failed' });
  }
});

router.get('/roles', async (_req, res) => {
  const rows = await listRolePermissions();
  const byRole = {};
  rows.forEach((row) => {
    if (!row?.role) return;
    const key = String(row.role).toLowerCase();
    byRole[key] = row.permissions || {};
  });
  const defaults = { ...DEFAULT_ROLE_PERMISSIONS };
  const baseRoles = Object.keys(DEFAULT_ROLE_PERMISSIONS).map((r) => r.toLowerCase());
  const allRoles = new Set([...baseRoles, ...Object.keys(byRole)]);
  const roles = Array.from(allRoles)
    .filter(Boolean)
    .sort((a, b) => a.localeCompare(b, 'es', { sensitivity: 'base' }))
    .map((role) => {
      const stored = byRole[role] || {};
      const effective = resolveRolePermissions(role, stored);
      const isCustom = !Object.prototype.hasOwnProperty.call(DEFAULT_ROLE_PERMISSIONS, role);
      if (isCustom && !defaults[role]) defaults[role] = {};
      return { role, effective, stored, isCustom };
    });
  res.json({ roles, defaults });
});

router.put('/roles/:role', async (req, res) => {
  try {
    const role = String(req.params.role || '').trim().toLowerCase();
    if (!role || !/^[a-z0-9_-]{3,32}$/.test(role)) {
      return res.status(400).json({ error: 'invalid_role' });
    }
    const rawPermissions = req.body?.permissions;
    if (rawPermissions !== undefined && (typeof rawPermissions !== 'object' || rawPermissions === null)) {
      return res.status(400).json({ error: 'invalid_permissions' });
    }
    const permissions = {};
    if (rawPermissions && typeof rawPermissions === 'object') {
      Object.entries(rawPermissions).forEach(([key, value]) => {
        permissions[key] = Boolean(value);
      });
    }
    await setRolePermissions(role, permissions);
    res.json({ ok: true });
  } catch (err) {
    console.error('[admin] set permissions error', err);
    res.status(500).json({ error: 'permissions_failed' });
  }
});

function generateAccountCode() {
  const alphabet = 'ABCDEFGHJKLMNPQRSTUVWXYZ23456789';
  let code = '';
  for (let i = 0; i < 6; i += 1) {
    code += alphabet[Math.floor(Math.random() * alphabet.length)];
  }
  return code;
}

export default router;
