import { Router } from 'express';
import { signJwt } from '../auth/jwt.js';
import { verifyPassword } from '../security/password.js';
import { getPanelAccountByUsername } from '../utils/db.js';

const router = Router();

function sanitizeAccount(account) {
  if (!account) return null;
  return {
    id: account.id,
    accountCode: account.accountCode || null,
    username: account.username,
    role: account.role,
    avatarUrl: account.avatarUrl || null,
    discordId: account.discordId || null,
    createdAt: account.createdAt || null,
    updatedAt: account.updatedAt || null,
  };
}

function setAuthCookie(res, token, req) {
  const sameSite = process.env.COOKIE_SAMESITE || 'lax';
  const secure = (process.env.COOKIE_SECURE || '').toLowerCase() === 'true' || req?.secure;
  const path = process.env.PUBLIC_BASE_PATH
    ? `/${String(process.env.PUBLIC_BASE_PATH).replace(/\/+$/, '').replace(/^\/+/, '')}`
    : '/';
  res.cookie('auth', token, {
    httpOnly: true,
    sameSite,
    secure,
    maxAge: 1000 * 60 * 60 * 24 * 7,
    path,
  });
}

router.post('/login', async (req, res) => {
  try {
    const { username, password } = req.body || {};
    if (typeof username !== 'string' || typeof password !== 'string') {
      return res.status(400).json({ error: 'invalid_credentials' });
    }
    const cleanUsername = username.trim();
    if (!cleanUsername || !password) {
      return res.status(400).json({ error: 'invalid_credentials' });
    }
    const account = await getPanelAccountByUsername(cleanUsername);
    if (!account || !account.password_hash) {
      return res.status(401).json({ error: 'invalid_credentials' });
    }
    const ok = await verifyPassword(password, account.password_hash);
    if (!ok) {
      return res.status(401).json({ error: 'invalid_credentials' });
    }
    const payload = {
      accountId: account.id,
      accountCode: account.accountCode || null,
      username: account.username,
      role: account.role,
      avatarUrl: account.avatarUrl || null,
      discordId: account.discordId || null,
      type: 'panel',
    };
    const token = signJwt(payload);
    setAuthCookie(res, token, req);
    return res.json({ ok: true, user: sanitizeAccount(account) });
  } catch (err) {
    console.error('[auth] login failed', err);
    return res.status(500).json({ error: 'login_failed' });
  }
});

router.post('/logout', (req, res) => {
  const sameSite = process.env.COOKIE_SAMESITE || 'lax';
  const secure = (process.env.COOKIE_SECURE || '').toLowerCase() === 'true' || req.secure;
  const path = process.env.PUBLIC_BASE_PATH
    ? `/${String(process.env.PUBLIC_BASE_PATH).replace(/\/+$/, '').replace(/^\/+/, '')}`
    : '/';
  res.clearCookie('auth', { httpOnly: true, sameSite, secure, path });
  res.json({ ok: true });
});

export default router;
