import { Router } from 'express';
import { requireAuth } from '../middlewares/requireAuth.js';
import { requireStaff } from '../middlewares/requireRole.js';
import { query } from '../utils/db.js';
const { SINGLE_GUILD_ID = null } = process.env;

const r = Router();

// Fetch the latest logs from the database. You can optionally filter by guild
// ID via the `x-guild-id` header. Returns up to 200 entries sorted by
// newest first. Each entry includes timestamp, origin, event summary and
// severity inferred from the payload.
r.get('/', requireAuth, requireStaff, async (req, res) => {
  try {
    const guildId = SINGLE_GUILD_ID || req.header('x-guild-id') || null;
    let limit = parseInt(req.query.limit || '20', 10);
    if (!Number.isFinite(limit) || limit <= 0) limit = 20;
    if (limit > 200) limit = 200;
    let rows;
    if (guildId) {
      rows = await query(
        'SELECT guildId, eventType, details, timestamp FROM detailed_logs WHERE guildId = ? ORDER BY timestamp DESC LIMIT ?',
        [guildId, limit]
      );
    } else {
      rows = await query(
        'SELECT guildId, eventType, details, timestamp FROM detailed_logs ORDER BY timestamp DESC LIMIT ?',
        [limit]
      );
    }
    // Map DB rows to API response
    const result = rows.map((row) => {
      let parsed;
      try {
        parsed = JSON.parse(row.details || '{}');
      } catch {
        parsed = {};
      }
      const ts = new Date(row.timestamp).toISOString();
      // Determine severity: use parsed.severity if provided, else infer
      let severity = parsed.severity || 'info';
      if (severity !== 'info' && severity !== 'warn' && severity !== 'error') severity = 'info';
      // Build event summary using origin and event/command fields
      const origin = parsed.origin || row.eventType;
      const event = parsed.event || parsed.command || parsed.description || '';
      return {
        ts,
        origin,
        event,
        severity,
      };
    });
    res.json(result);
  } catch (err) {
    console.error('[logs] Error fetching logs:', err);
    res.status(500).json({ error: 'failed_to_fetch_logs' });
  }
});

export default r;
