import bcrypt from 'bcryptjs';

const SALT_ROUNDS = 12;

/**
 * Hashes a plain text password using bcrypt with the configured cost factor.
 * @param {string} password
 * @returns {Promise<string>}
 */
export async function hashPassword(password) {
  if (typeof password !== 'string' || !password) {
    throw new Error('invalid_password');
  }
  return bcrypt.hash(password, SALT_ROUNDS);
}

/**
 * Compares a plain text password against a previously hashed password.
 * @param {string} password
 * @param {string} hash
 * @returns {Promise<boolean>}
 */
export async function verifyPassword(password, hash) {
  if (!hash) return false;
  if (typeof password !== 'string' || !password) return false;
  try {
    return await bcrypt.compare(password, hash);
  } catch {
    return false;
  }
}
