// panel/src/utils/db.js
// Utility for connecting to MySQL and ensuring required tables exist.
// This file reuses the same MySQL database as the Discord bot (firebot).
// It reads connection settings from environment variables and exports helper
// functions to initialize the connection and run queries.

import 'dotenv/config';
import mysql from 'mysql2/promise';

// Read DB configuration from environment variables. These should match the
// settings used by your bot so both panel and bot share the same database.
const {
  DB_HOST = '69.48.202.175',
  DB_PORT = '3306',
  DB_USER = 'admin',
  DB_PASSWORD = 'passSegura33',
  DB_NAME = 'firebot',
  DB_CONN_LIMIT = '10',
  DB_TIMEZONE = 'Z',
  DB_DATE_STRINGS = 'true',
  DB_CREATE_IF_MISSING = 'true',
} = process.env;

let pool;

/**
 * Initializes the MySQL connection pool and creates the database/table if needed.
 */
export async function initialize() {
  if (pool) return;
  // Optionally create the database if missing
  if (DB_CREATE_IF_MISSING === 'true') {
    const conn = await mysql.createConnection({
      host: DB_HOST,
      port: Number(DB_PORT),
      user: DB_USER,
      password: DB_PASSWORD,
      timezone: DB_TIMEZONE,
      dateStrings: DB_DATE_STRINGS === 'true',
    });
    await conn.query(`CREATE DATABASE IF NOT EXISTS \`${DB_NAME}\``);
    await conn.end();
  }
  // Create connection pool
  pool = mysql.createPool({
    host: DB_HOST,
    port: Number(DB_PORT),
    user: DB_USER,
    password: DB_PASSWORD,
    database: DB_NAME,
    waitForConnections: true,
    connectionLimit: Number(DB_CONN_LIMIT),
    timezone: DB_TIMEZONE,
    dateStrings: DB_DATE_STRINGS === 'true',
  });
  // Ensure the detailed_logs table exists
  await pool.query(`CREATE TABLE IF NOT EXISTS detailed_logs (
    id INT AUTO_INCREMENT PRIMARY KEY,
    guildId VARCHAR(20) NULL,
    userId VARCHAR(20) NULL,
    eventType VARCHAR(255) NOT NULL,
    details JSON NULL,
    timestamp BIGINT NOT NULL,
    INDEX idx_timestamp (timestamp),
    INDEX idx_guild (guildId)
  ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4`);

  // Ensure the user_ranks table exists (for role/permissions)
  await pool.query(`CREATE TABLE IF NOT EXISTS user_ranks (
    userId VARCHAR(20) NOT NULL,
    guildId VARCHAR(20) NULL,
    \`rank\` VARCHAR(32) NOT NULL,
    UNIQUE KEY uniq_user_guild (userId, guildId),
    INDEX idx_user (userId),
    INDEX idx_guild (guildId)
  ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4`);

  // Tickets table to track open/closed tickets from the bot
  await pool.query(`CREATE TABLE IF NOT EXISTS tickets (
    channelId VARCHAR(32) PRIMARY KEY,
    guildId   VARCHAR(32) NOT NULL,
    openerId  VARCHAR(32) DEFAULT NULL,
    staffId   VARCHAR(32) DEFAULT NULL,
    category  VARCHAR(32) DEFAULT NULL,
    status    VARCHAR(16) NOT NULL DEFAULT 'open',
    createdAt BIGINT DEFAULT NULL,
    updatedAt BIGINT NOT NULL
  ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4`);

  // Panel users: map Discord user to persistent numeric panelId
  await pool.query(`CREATE TABLE IF NOT EXISTS panel_users (
    discordId VARCHAR(32) NOT NULL,
    panelId INT UNSIGNED NOT NULL,
    createdAt BIGINT NOT NULL,
    PRIMARY KEY (discordId),
    UNIQUE KEY uniq_panelId (panelId)
  ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4`);
  // Optional columns to store profile info
  try {
    await pool.query(`ALTER TABLE panel_users
      ADD COLUMN IF NOT EXISTS username VARCHAR(100) NULL,
      ADD COLUMN IF NOT EXISTS avatar VARCHAR(128) NULL,
      ADD COLUMN IF NOT EXISTS updatedAt BIGINT NOT NULL DEFAULT 0,
      ADD COLUMN IF NOT EXISTS discordRoleName VARCHAR(100) NULL,
      ADD COLUMN IF NOT EXISTS discordRoleColor INT NULL`);
  } catch {}
  // Panel accounts for native login
  await pool.query(`CREATE TABLE IF NOT EXISTS panel_accounts (
    id INT UNSIGNED NOT NULL AUTO_INCREMENT,
    accountCode VARCHAR(32) NOT NULL,
    username VARCHAR(64) NOT NULL,
    password_hash VARCHAR(255) NOT NULL,
    role VARCHAR(32) NOT NULL,
    avatarUrl VARCHAR(255) NULL,
    discordId VARCHAR(32) NULL,
    createdAt BIGINT NOT NULL,
    updatedAt BIGINT NOT NULL,
    PRIMARY KEY (id),
    UNIQUE KEY uniq_panel_code (accountCode),
    UNIQUE KEY uniq_panel_user (username),
    UNIQUE KEY uniq_panel_discord (discordId)
  ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4`);
  try {
    await pool.query(`ALTER TABLE panel_accounts
      ADD COLUMN IF NOT EXISTS accountCode VARCHAR(32) NOT NULL,
      ADD UNIQUE INDEX IF NOT EXISTS uniq_panel_code (accountCode)`);
  } catch {}
  await pool.query(`CREATE TABLE IF NOT EXISTS panel_role_permissions (
    role VARCHAR(32) NOT NULL PRIMARY KEY,
    permissions JSON NOT NULL,
    updatedAt BIGINT NOT NULL
  ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4`);
}

/**
 * Executes a SQL query. Returns the result rows.
 * @param {string} sql The SQL query with placeholders.
 * @param {any[]} params The parameters for the query.
 */
export async function query(sql, params = []) {
  const [rows] = await pool.query(sql, params);
  return rows;
}

/**
 * Runs a query and returns a single row (or undefined).
 */
export async function get(sql, params = []) {
  const rows = await query(sql, params);
  return rows[0];
}

/**
 * Runs a query and returns all rows.
 */
export async function all(sql, params = []) {
  return query(sql, params);
}

/**
 * Closes the database pool.
 */
export async function close() {
  if (pool) {
    await pool.end();
    pool = null;
  }
}

export async function getUserRank(userId, guildId) {
  const rows = await query(
    'SELECT `rank` FROM user_ranks WHERE userId = ? AND (guildId IS NULL OR guildId = ?) ORDER BY guildId DESC LIMIT 1',
    [userId, guildId]
  );
  return rows[0]?.rank ?? null;
}

export async function setUserRank(userId, guildId, rank) {
  await query(
    'INSERT INTO user_ranks (userId, guildId, `rank`) VALUES (?,?,?) ON DUPLICATE KEY UPDATE `rank` = VALUES(`rank`)',
    [userId, guildId, rank]
  );
}

/**
 * Returns an existing panelId for the given Discord user, or creates a new
 * unique numeric panelId on first call.
 */
export async function getOrCreatePanelId(discordId) {
  if (!discordId) throw new Error('missing_discord_id');
  const existing = await get('SELECT panelId FROM panel_users WHERE discordId = ? LIMIT 1', [discordId]);
  if (existing?.panelId != null) return existing.panelId;
  // Try to insert with random unique panelId, retry on collision
  for (let i = 0; i < 10; i++) {
    // 8-digit number [10,000,000 - 99,999,999]
    const candidate = 10_000_000 + Math.floor(Math.random() * 90_000_000);
    try {
      await query(
        'INSERT INTO panel_users (discordId, panelId, createdAt) VALUES (?, ?, ?)',
        [discordId, candidate, Date.now()]
      );
      return candidate;
    } catch (e) {
      if (e?.code === 'ER_DUP_ENTRY') continue; // collided, try again
      throw e;
    }
  }
  // As a last resort, read again in case another process inserted concurrently
  const row = await get('SELECT panelId FROM panel_users WHERE discordId = ? LIMIT 1', [discordId]);
  if (row?.panelId != null) return row.panelId;
  throw new Error('failed_to_assign_panel_id');
}

export async function updatePanelUserInfo(discordId, { username = null, avatar = null } = {}) {
  if (!discordId) return;
  try {
    await query(
      'UPDATE panel_users SET username = COALESCE(?, username), avatar = COALESCE(?, avatar), updatedAt = ? WHERE discordId = ?',
      [username, avatar, Date.now(), discordId]
    );
  } catch {}
}

export async function updatePanelUserDiscordRole(discordId, { roleName = null, roleColor = null } = {}) {
  if (!discordId) return;
  try {
    await query(
      'UPDATE panel_users SET discordRoleName = COALESCE(?, discordRoleName), discordRoleColor = COALESCE(?, discordRoleColor), updatedAt = ? WHERE discordId = ?',
      [roleName, roleColor, Date.now(), discordId]
    );
  } catch {}
}

export async function createPanelAccount({ accountCode, username, passwordHash, role, avatarUrl = null, discordId = null }) {
  if (!accountCode) throw new Error('missing_account_code');
  if (!username) throw new Error('missing_username');
  if (!passwordHash) throw new Error('missing_password_hash');
  if (!role) throw new Error('missing_role');
  const now = Date.now();
  const [result] = await pool.query(
    'INSERT INTO panel_accounts (accountCode, username, password_hash, role, avatarUrl, discordId, createdAt, updatedAt) VALUES (?, ?, ?, ?, ?, ?, ?, ?)',
    [accountCode, username, passwordHash, role, avatarUrl, discordId, now, now]
  );
  return { id: result.insertId, createdAt: now };
}

export async function updatePanelAccount(id, fields = {}) {
  if (!id) throw new Error('missing_id');
  const allowed = { accountCode: 'accountCode', username: 'username', role: 'role', avatarUrl: 'avatarUrl', discordId: 'discordId' };
  const updates = [];
  const params = [];
  for (const [key, value] of Object.entries(fields)) {
    if (!(key in allowed) || value === undefined) continue;
    updates.push(`${allowed[key]} = ?`);
    params.push(value);
  }
  if (!updates.length) return false;
  const now = Date.now();
  updates.push('updatedAt = ?');
  params.push(now, id);
  const sql = `UPDATE panel_accounts SET ${updates.join(', ')} WHERE id = ?`;
  const [result] = await pool.query(sql, params);
  return result.affectedRows > 0;
}

export async function setPanelAccountPassword(id, passwordHash) {
  if (!id) throw new Error('missing_id');
  if (!passwordHash) throw new Error('missing_password_hash');
  const now = Date.now();
  const [result] = await pool.query(
    'UPDATE panel_accounts SET password_hash = ?, updatedAt = ? WHERE id = ?',
    [passwordHash, now, id]
  );
  return result.affectedRows > 0;
}

export async function getPanelAccountByUsername(username) {
  if (!username) return null;
  return await get('SELECT * FROM panel_accounts WHERE username = ? LIMIT 1', [username]);
}

export async function getPanelAccountById(id) {
  if (!id) return null;
  return await get('SELECT * FROM panel_accounts WHERE id = ? LIMIT 1', [id]);
}

export async function getPanelAccountByDiscordId(discordId) {
  if (!discordId) return null;
  return await get('SELECT * FROM panel_accounts WHERE discordId = ? LIMIT 1', [discordId]);
}

export async function listPanelAccounts() {
  return await query('SELECT id, accountCode, username, role, avatarUrl, discordId, createdAt, updatedAt FROM panel_accounts ORDER BY username', []);
}

export async function unlinkPanelAccountDiscord(id) {
  if (!id) throw new Error('missing_id');
  const now = Date.now();
  const [result] = await pool.query(
    'UPDATE panel_accounts SET discordId = NULL, updatedAt = ? WHERE id = ?',
    [now, id]
  );
  return result.affectedRows > 0;
}

export async function linkPanelAccountDiscord(id, discordId) {
  if (!id) throw new Error('missing_id');
  if (!discordId) throw new Error('missing_discord_id');
  const now = Date.now();
  const [result] = await pool.query(
    'UPDATE panel_accounts SET discordId = ?, updatedAt = ? WHERE id = ?',
    [discordId, now, id]
  );
  return result.affectedRows > 0;
}

export async function getRolePermissions(role) {
  if (!role) return null;
  const row = await get('SELECT role, permissions, updatedAt FROM panel_role_permissions WHERE role = ? LIMIT 1', [role]);
  if (!row) return null;
  let perms = {};
  try { perms = row.permissions ? JSON.parse(row.permissions) : {}; } catch { perms = {}; }
  return { role: row.role, permissions: perms, updatedAt: row.updatedAt };
}

export async function setRolePermissions(role, permissions = {}) {
  if (!role) throw new Error('missing_role');
  const now = Date.now();
  const payload = JSON.stringify(permissions ?? {});
  await pool.query(
    `INSERT INTO panel_role_permissions (role, permissions, updatedAt)
     VALUES (?, ?, ?)
     ON DUPLICATE KEY UPDATE
       permissions = VALUES(permissions),
       updatedAt = VALUES(updatedAt)`,
    [role, payload, now]
  );
}

export async function listRolePermissions() {
  const rows = await query('SELECT role, permissions, updatedAt FROM panel_role_permissions', []);
  return rows.map(row => {
    let perms = {};
    try { perms = row.permissions ? JSON.parse(row.permissions) : {}; } catch { perms = {}; }
    return { role: row.role, permissions: perms, updatedAt: row.updatedAt };
  });
}
